﻿// An F# console application with benchmarks        Michael R. Hansen 12-11-2024
// -- with and without Ahead Of Time compilation 

// The projected was created as follows:
// dotnet new console -lang F#
// dotnet build 
// --- makes an executable in the bin\Debug\net8.0 directory

// Native code can be created as follows: 
// dotnet publish -r win-x64 -c Release
// --- makes an executable in the \bin\Release\net8.0\win-x64 directory

// observe the different runtimes below


// Three versions of the factorial function:

let rec fact = function
               | 0 -> 1
               | n -> n * fact(n-1)

let rec factA(n,m) = match n with  
                     | 0 -> m
                     | _ -> factA(n-1,n*m) 

let factW n = let ni = ref n
              let r  = ref 1
              while ni.Value>0 do
                  r.Value <- r.Value * ni.Value ; 
                  ni.Value <- ni.Value-1
              r.Value

let rec naiveRev = 
   function
   | []    -> []
   | x::xs -> naiveRev xs @ [x]

let rec revA = 
   function
   | ([], ys)    -> ys
   | (x::xs, ys) -> revA(xs, x::ys)


// measurement of time using a stopwatch 
let duration f a = 
    let timer = System.Diagnostics.Stopwatch.StartNew()
    timer.Start()
    let _ = f a
    let _ = timer.Stop()
    float timer.ElapsedMilliseconds

let mean ns = List.foldBack (+) ns 0.0 / float(List.length ns)
let stdDev ns m = let n = float(List.length ns) 
                  System.Math.Round(sqrt((List.foldBack (fun n acc -> (n - m)**2.0 + acc) ns 0.0) / n),2) 

// makes a report concerning 'times' repeated executions of 'f(arg)'.   
let reportTiming f arg times txt = 
   let ns = [ for i in 1..times -> duration f arg ]
   let m = mean ns
   $"""{txt} are repeated {times} times. Mean = {m} ms. StdDev = {stdDev ns m}. Min = {List.min ns}. Max = {List.max ns}"""  
// using an interpolated raw string


let xs16 = List.init 1000000 (fun i -> 16) 

let factW16() = for i in xs16 do let _ = factW i in () 
let fact16() = for i in xs16 do let _ = fact i in () 
let factA16() = for i in xs16 do let _ = factA(i,1) in ()


let xs500000 = [1 .. 500000]
let xs5000 = [1 .. 5000]

   
printfn "%s" (reportTiming fact16 () 100  "1.000000 computations of fact 16")
printfn "%s" (reportTiming factA16 () 100 "1.000000 computations of factA(16,1)")     
printfn "%s" (reportTiming factW16 () 100  "1.000000 computations of factW 16")

printfn "%s" "\n"

printfn "%s" (reportTiming naiveRev xs5000 100 "naive reverse of a list of size 5.000")
printfn "%s" (reportTiming revA (xs500000,[]) 100 "tail-recursive reverse of a list of size 500.000") 

ignore (System.Console.ReadKey())

(* run in "Debug mode"  (See e.g. https://learn.microsoft.com/en-us/dotnet/standard/managed-execution-process)
--- At execution time: A JIT (Just-In-Time) compiler translates intermediate code to native (CPU specific) code.

C:\...\Kurser\02157-24\Benchmarks>dotnet run
1.000000 computations of fact 16 are repeated 100 times. Mean = 37,78ms. StdDev = 3,8. Min = 33. Max = 51
1.000000 computations of factA(16,1) are repeated 100 times. Mean = 25ms. StdDev = 1,75. Min = 22. Max = 30
1.000000 computations of factW 16 are repeated 100 times. Mean = 129,67ms. StdDev = 19,23. Min = 113. Max = 208


naive reverse of a list of size 5.000 are repeated 100 times. Mean = 133,47ms. StdDev = 18,74. Min = 111. Max = 219
tail-recursive reverse of a list of size 500.000 are repeated 100 times. Mean = 54,13ms. StdDev = 18,97. Min = 33. Max = 102
*)


(* Execution in "Release mode": (see e.g. https://learn.microsoft.com/en-us/dotnet/core/deploying/native-aot/?tabs=windows%2Cnet8)
--- Benchmarks.exe is constructed using AOT (Ahead-Of-Time) compilation 
--- produces native win-x64 code (in this case)
--- a JIT compiler is NOT used when the code executes

C:\...\Kurser\02157-24\Benchmarks\bin\Release\net8.0\win-x64>Benchmarks.exe
1.000000 computations of fact 16 are repeated 100 times. Mean = 15,13ms. StdDev = 2,35. Min = 13. Max = 24
1.000000 computations of factA(16,1) are repeated 100 times. Mean = 6,89ms. StdDev = 7,61. Min = 5. Max = 45
1.000000 computations of factW 16 are repeated 100 times. Mean = 26,63ms. StdDev = 12,74. Min = 23. Max = 133


naive reverse of a list of size 5.000 are repeated 100 times. Mean = 88,72ms. StdDev = 14,01. Min = 71. Max = 138
tail-recursive reverse of a list of size 500.000 are repeated 100 times. Mean = 24,29ms. StdDev = 15,27. Min = 12. Max = 82
*)


